---
name: bogostian-ontographer
description: Transform messy text into ontographically rich word lists using Ian Bogost's Alien Phenomenology principles. Creates JavaScript wordPool arrays identical to tug.js format for litany generators. Use when extracting evocative word pools from documents, web scrapes, or URLs for philosophical litany generators, tiny ontology machines, or exploring flat ontology through language.
---

# Bogostian Ontographer

Transform any text into an ontographically rich JavaScript word pool following Ian Bogost's principles from "Alien Phenomenology."

## Primary Output: JavaScript Word Pools

This skill specializes in creating `.js` files formatted exactly like `tug.js`:

```javascript
const wordPool = [
    "lighthouse",
    "carbon", 
    "dragonfly",
    "tornado",
    "vinyl",
    "consciousness",
    // ... hundreds more ontologically rich words
];
```

## Quick Start

### One-Command Pipeline

```bash
# Transform any text directly to wordPool.js
python3 scripts/text_to_js.py input.txt

# From a URL
curl -s "https://example.com/page" | python3 scripts/text_to_js.py -

# With custom name and sorting (like tug.js)
python3 scripts/text_to_js.py input.txt -o myWords.js --sort

# Adjust pool size
python3 scripts/text_to_js.py input.txt --max-words 1000
```

## The Bogostian Process

The skill applies a 6-stage philosophical filtering paradigm:

1. **Technical Cleaning**: Deduplication, normalization, encoding fixes
2. **Enjambment Separation**: Smart splitting of compound words  
3. **Dictionary Validation**: Real word verification, proper noun detection
4. **Ontological Filtration**: Apply Bogost's flat ontology criteria
5. **Morphological Deduplication**: Choose most evocative word forms
6. **Enrichment Check**: Ensure heterogeneity and mutual alienness

## Advanced Usage

### Step-by-Step Control

For fine-grained control, run the cleaning and JS generation separately:

```bash
# Step 1: Clean with detailed statistics
python3 scripts/bogostian_clean.py input.txt --output-format json > cleaned.json

# Step 2: Generate JavaScript 
python3 scripts/generate_js.py cleaned.json -o wordPool.js --sort
```

### Multiple Input Formats

The cleaner accepts various inputs:

```bash
# From file
python3 scripts/text_to_js.py document.txt

# From stdin
cat document.txt | python3 scripts/text_to_js.py -

# From URL (via curl)
curl -s "https://site.com" | python3 scripts/text_to_js.py - -o site-words.js

# From multiple files (combine first)
cat *.txt | python3 scripts/text_to_js.py - -o combined.js
```

### Output Options

```bash
# Just cleaning (multiple formats)
python3 scripts/bogostian_clean.py input.txt --output-format json    # Full analysis
python3 scripts/bogostian_clean.py input.txt --output-format list    # Simple list
python3 scripts/bogostian_clean.py input.txt --output-format javascript  # JS array

# JS generation options
python3 scripts/generate_js.py words.json --sort              # Alphabetical
python3 scripts/generate_js.py words.json --capitalize-proper # Fix capitalization
```

## Quality Criteria

A good Bogostian word pool exhibits:

✓ **Heterogeneity**: Objects at wildly different scales  
✓ **Flatness**: No hierarchical relationships  
✓ **Disjunction**: Words resist narrative flow  
✓ **Density**: Each word suggests hidden depths  
✓ **Size**: 30-500 words optimal (100-200 ideal)

## Integration with Litany Generators

Once you have your wordPool.js:

```javascript
// Include in your HTML
<script src="wordPool.js"></script>

// Use in your generator
function generateLitany(count = 7) {
    const selected = [];
    const pool = [...wordPool];
    
    for (let i = 0; i < count && pool.length > 0; i++) {
        const index = Math.floor(Math.random() * pool.length);
        selected.push(pool.splice(index, 1)[0]);
    }
    
    return selected.join(', ');
}

// Output: "lighthouse, quantum, vinyl, cathedral, piranha, consciousness, salt"
```

## Examples of Good Sources

- **Literary texts**: Rich vocabulary, varied scales
- **Encyclopedia articles**: Wide range of concrete references  
- **Technical manuals**: Objects, materials, processes
- **Historical documents**: Places, people, artifacts
- **Web scrapes**: After heavy filtering

## Troubleshooting

**Too few words?**
- Try more content-rich sources
- Lower minimum threshold: `--min-words 20`
- Check if source is too abstract/interface-heavy

**Words too similar?**
- Source may lack diversity
- Combine multiple sources
- Adjust morphological deduplication in script

**Want exact tug.js format?**
- Use `--sort` flag for alphabetical ordering
- Default output matches format exactly

## Philosophical Context

This tool implements Bogost's "flat ontology" where all things exist equally. The goal is creating what Bogost calls "the jarring staccato of real being" - not smooth taxonomy but surprising juxtaposition of "mutual aliens."

For deeper understanding of the principles, see `references/paradigm.md`.

## Scripts Included

- `text_to_js.py` - Complete pipeline (text → JS)  
- `bogostian_clean.py` - Core cleaning with 6-stage filtering
- `generate_js.py` - Convert word lists to JS format
- `references/paradigm.md` - Complete philosophical framework

Ready to transform your texts into ontographically rich JavaScript word pools!
