#!/usr/bin/env python3
"""
Generate a JavaScript word pool file in the tug.js format
Outputs a formatted .js file with const wordPool array
"""

import json
import sys
import argparse

def format_js_wordpool(words):
    """Format words as a JavaScript const array like tug.js."""
    
    # Start the JavaScript const declaration
    output = "const wordPool = [\n"
    
    # Add each word as a quoted string with proper indentation
    for i, word in enumerate(words):
        # Escape any quotes in the word
        escaped_word = word.replace('"', '\\"')
        
        # Add the word with 4-space indentation
        output += f'    "{escaped_word}"'
        
        # Add comma if not the last word
        if i < len(words) - 1:
            output += ","
        
        output += "\n"
    
    # Close the array
    output += "];\n"
    
    return output

def main():
    parser = argparse.ArgumentParser(
        description='Generate a JavaScript word pool file from cleaned words'
    )
    parser.add_argument('input', help='Input JSON file from bogostian_clean.py or text file with one word per line')
    parser.add_argument('-o', '--output', help='Output .js file (default: wordPool.js)', 
                       default='wordPool.js')
    parser.add_argument('--sort', action='store_true', 
                       help='Sort words alphabetically (case-insensitive)')
    parser.add_argument('--capitalize-proper', action='store_true',
                       help='Ensure proper nouns are capitalized')
    
    args = parser.parse_args()
    
    # Read input
    with open(args.input, 'r', encoding='utf-8') as f:
        content = f.read()
    
    # Try to parse as JSON first
    try:
        data = json.loads(content)
        if isinstance(data, dict) and 'words' in data:
            # It's output from bogostian_clean.py
            words = data['words']
        elif isinstance(data, list):
            # It's already a list of words
            words = data
        else:
            raise ValueError("Unexpected JSON format")
    except (json.JSONDecodeError, ValueError):
        # Treat as plain text, one word per line
        words = [line.strip() for line in content.split('\n') if line.strip()]
    
    # Process words
    if args.capitalize_proper:
        # Simple heuristic: capitalize if it looks like a proper noun
        processed_words = []
        for word in words:
            # Keep existing capitalization for already capitalized words
            # Otherwise use lowercase
            if word and word[0].isupper():
                processed_words.append(word)
            else:
                processed_words.append(word.lower())
        words = processed_words
    
    # Sort if requested
    if args.sort:
        # Sort case-insensitively but preserve original capitalization
        words = sorted(words, key=lambda w: w.upper())
    
    # Generate JavaScript output
    js_content = format_js_wordpool(words)
    
    # Write output file
    with open(args.output, 'w', encoding='utf-8') as f:
        f.write(js_content)
    
    print(f"✓ Generated {args.output} with {len(words)} words")
    
    # Print sample if small enough
    if len(words) <= 10:
        print(f"Words: {', '.join(words)}")
    else:
        print(f"Sample: {', '.join(words[:5])}... and {len(words)-5} more")

if __name__ == '__main__':
    main()
