#!/usr/bin/env python3
"""
Complete pipeline: Text → Bogostian Cleaning → JavaScript wordPool.js
Combines bogostian_clean.py and generate_js.py into one workflow
"""

import subprocess
import tempfile
import json
import sys
import argparse
import os

def run_pipeline(input_file, output_file='wordPool.js', min_words=30, max_words=500, sort_words=False):
    """Run the complete Bogostian cleaning to JS pipeline."""
    
    # Create temporary file for intermediate JSON
    with tempfile.NamedTemporaryFile(mode='w', suffix='.json', delete=False) as tmp:
        tmp_json = tmp.name
    
    try:
        print("🔄 Starting Bogostian ontographical processing...")
        print("-" * 50)
        
        # Step 1: Run Bogostian cleaning
        print("Step 1: Applying philosophical filtering...")
        clean_cmd = [
            sys.executable, 
            os.path.join(os.path.dirname(__file__), 'bogostian_clean.py'),
            input_file,
            '--output-format', 'json',
            '--min-words', str(min_words),
            '--max-words', str(max_words)
        ]
        
        result = subprocess.run(clean_cmd, capture_output=True, text=True)
        
        if result.returncode != 0:
            print(f"Error in cleaning: {result.stderr}", file=sys.stderr)
            return False
        
        # Save JSON output
        with open(tmp_json, 'w') as f:
            f.write(result.stdout)
        
        # Parse to get statistics
        data = json.loads(result.stdout)
        word_count = data['count']
        categories = data['categories']
        
        print(f"✓ Extracted {word_count} ontologically rich words")
        print(f"  Categories: {', '.join(f'{k}={v}' for k, v in categories.items() if v > 0)}")
        print()
        
        # Step 2: Generate JavaScript file
        print("Step 2: Generating JavaScript wordPool...")
        js_cmd = [
            sys.executable,
            os.path.join(os.path.dirname(__file__), 'generate_js.py'),
            tmp_json,
            '-o', output_file
        ]
        
        if sort_words:
            js_cmd.append('--sort')
            
        result = subprocess.run(js_cmd, capture_output=True, text=True)
        
        if result.returncode != 0:
            print(f"Error generating JS: {result.stderr}", file=sys.stderr)
            return False
        
        print(f"✓ Generated {output_file}")
        print()
        
        # Show sample of words
        if word_count > 0:
            sample_size = min(7, word_count)
            sample = data['words'][:sample_size]
            print("🎲 Sample litany from your word pool:")
            print(f"   \"{', '.join(sample)}\"")
            if word_count > sample_size:
                print(f"   ...and {word_count - sample_size} more words")
        
        print("-" * 50)
        print(f"✨ Complete! Your Bogostian word pool is in {output_file}")
        print(f"   Ready for use in litany generators and ontology machines")
        
        return True
        
    finally:
        # Clean up temporary file
        if os.path.exists(tmp_json):
            os.remove(tmp_json)

def main():
    parser = argparse.ArgumentParser(
        description='Complete pipeline: Text → Bogostian Cleaning → JavaScript wordPool',
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  %(prog)s document.txt                  # Creates wordPool.js
  %(prog)s document.txt -o myWords.js    # Custom output name
  %(prog)s document.txt --sort           # Alphabetically sorted
  %(prog)s document.txt --max-words 1000 # Larger word pool
  
  # From URL:
  curl -s "https://example.com/text" | %(prog)s - -o site-words.js
"""
    )
    
    parser.add_argument('input', 
                       help='Input file path (or - for stdin)')
    parser.add_argument('-o', '--output', 
                       default='wordPool.js',
                       help='Output JavaScript file (default: wordPool.js)')
    parser.add_argument('--min-words', 
                       type=int, default=30,
                       help='Minimum words for good litany (default: 30)')
    parser.add_argument('--max-words', 
                       type=int, default=500,
                       help='Maximum words before truncation (default: 500)')
    parser.add_argument('--sort', 
                       action='store_true',
                       help='Sort words alphabetically in output')
    
    args = parser.parse_args()
    
    # Handle stdin
    if args.input == '-':
        with tempfile.NamedTemporaryFile(mode='w', suffix='.txt', delete=False) as tmp:
            tmp.write(sys.stdin.read())
            input_file = tmp.name
    else:
        input_file = args.input
    
    try:
        success = run_pipeline(
            input_file,
            args.output,
            args.min_words,
            args.max_words,
            args.sort
        )
        
        sys.exit(0 if success else 1)
        
    finally:
        # Clean up stdin temp file
        if args.input == '-' and 'input_file' in locals():
            if os.path.exists(input_file):
                os.remove(input_file)

if __name__ == '__main__':
    main()
